<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Lab;
use App\Models\Test;
use App\Http\Resources\Lab\LabResource;
use App\Http\Resources\Lab\LabTestCategoryResource;
use App\Http\Resources\Lab\StiTestResultResource;
use App\Http\Resources\Lab\LabTestHistoryResource;
use App\Http\Resources\Lab\QrCodeDataResource;
use App\Http\Requests\Lab\CustomerLabTest;
use App\Models\LabTestUser;
use App\Traits\ApiResponseTrait;
use App\Http\Resources\Lab\CustomerTestHistoryResource;
use App\Models\Customer;
use App\Models\StiTestResult;
use Illuminate\Support\Facades\DB;

class LabController extends Controller
{
    use ApiResponseTrait;
    public function index(Request $request)
    {
        $labs = Lab::select('id', 'name', 'lab_logo_url')->get();
        return $this->successResponse(LabResource::collection($labs), 'Labs fetched successfully.');
    }
    public function getTest(Request $request)
    {
        $request->validate([
            'lab_id' => 'required'
        ]);

        $test = DB::table('lab_test_users as ltu')
            ->select(
                DB::raw('MIN(ltu.lab_test_user_id) as id'),
                'ltu.lab_id',
                'labs.name as lab_name',
                DB::raw('MIN(ltu.lab_test_id) as lab_test_id'),
                'lab_tests.category_id',
                'tc.name as category_name'
            )
            ->leftJoin('labs', 'labs.id', '=', 'ltu.lab_id')
            ->leftJoin('customers', 'customers.id', '=', 'ltu.customer_id')
            ->leftJoin('lab_tests', 'lab_tests.id', '=', 'ltu.lab_test_id')
            ->leftJoin('test_categories as tc', 'tc.id', '=', 'lab_tests.category_id')
            ->where('ltu.lab_id', $request->lab_id)
            ->groupBy('tc.id', 'ltu.lab_id', 'labs.name', 'lab_tests.category_id', 'tc.name')
            ->get();
        
        return $this->successResponse(LabTestCategoryResource::collection($test), 'Tests fetched successfully.');
    }
    public function submitTest(CustomerLabTest $request)
    {
        DB::beginTransaction();

        try {

            $labtestUser = LabTestUser::create([
                'lab_id' => $request->laboratory,
                'lab_test_id' => $request->test_id,
                'patient_id' => $request->patientId,
                'test_date' => $request->testDate,
                'customer_id' => auth()->id(),
            ]);

            StiTestResult::create([
                'lab_test_user_id' => $labtestUser->lab_test_user_id,
            ]);

            DB::commit();

            return $this->successResponse([], 'Lab test submitted successfully.');
        } catch (\Exception $e) {

            DB::rollBack();

            return $this->errorResponse($e->getMessage(), 500);
        }
    }
    public function getLabTestHistory(Request $request)
    {
        $limit = $request->get('limit', 10);
        $offset = $request->get('offset', 0);
        $customer_id = auth()->id();

        // Get total count
        $total = DB::table('lab_test_results as r')
            ->leftJoin('lab_test_users as ltu', 'ltu.lab_test_user_id', '=', 'r.lab_test_user_id')
            ->leftJoin('customers', 'customers.id', '=', 'ltu.customer_id')
            ->whereNull('r.deleted_at')
            ->where('customers.id', $customer_id)
            ->count();

        // Get results using limit and offset
        $results = DB::table('lab_test_results as r')
            ->select(
                'r.id',
                'r.result_document_url',
                'r.status',
                'r.outcome',
                'r.notes',
                'r.created_at',
                'r.updated_at',
                'ltu.lab_test_user_id',
                'labs.id as lab_id',
                'labs.name as lab_name',
                'customers.id as customer_id',
                 DB::raw("CONCAT(customers.first_name, ' ', customers.last_name) AS customer_name"),
                'customers.email as customer_email',
                'ltu.patient_id',
                'ltu.test_date',
                'lab_tests.id as lab_test_id',
                'lab_tests.name as test_name',
                'lab_tests.test_type',
                'tc.id as category_id',
                'tc.name as category_name'
            )
            ->leftJoin('lab_test_users as ltu', 'ltu.lab_test_user_id', '=', 'r.lab_test_user_id')
            ->leftJoin('labs', 'labs.id', '=', 'ltu.lab_id')
            ->leftJoin('customers', 'customers.id', '=', 'ltu.customer_id')
            ->leftJoin('lab_tests', 'lab_tests.id', '=', 'ltu.lab_test_id')
            ->leftJoin('test_categories as tc', 'tc.id', '=', 'lab_tests.category_id')
            ->whereNull('r.deleted_at')
            ->where('customers.id', $customer_id)
            ->orderBy('r.created_at', 'DESC')
            ->limit($limit)
            ->offset($offset)
            ->get();

        // Build Response with limit/offset pagination
        $result = [
            'items' => LabTestHistoryResource::collection($results),
            'total' => $total,
            'limit' => (int)$limit,
            'offset' => (int)$offset,
        ];

        return $this->successResponse($result, 'Lab test history fetched successfully.');
    }
    public function stiDashboard()
    {

        $customer = Customer::select(
            'id',
            'first_name',
            'last_name',
            'profile_image_url',
            'email',
            'dob',
            'city',
            'gender_id',
            'email_verified_at',
            'mobile_verified'
        )
            ->with('gender:id,name')
            ->with([
                'labTestUsers' => function ($query) {
                    $query->select('lab_test_user_id as id', 'test_date', 'patient_id', 'customer_id', 'lab_id', 'lab_test_id');
                },
                'labTestUsers.lab' => function ($q) {
                    $q->select('id', 'name');
                }
            ])   // Load only id and name from genders table
            ->where('id', auth()->id())
            ->first();
        $labTests = LabTestUser::where('customer_id', $customer->id)->orderBy('test_date', 'DESC')->first();
        if ($labTests) {
            $testResult = StiTestResult::select('outcome')->where('lab_test_user_id', $labTests->lab_test_user_id)->first();
        }

        $customer->testStatus = isset($testResult->outcome) ? $testResult->outcome : 'positive';
        return $this->successResponse($customer, 'STI Dashboard data fetched successfully.');
    }
    public function qrCodeData()
    {
        $customer_id = auth()->id();
        $customerStiTest = LabTestUser::select('lab_test_id as id', 'customer_id')
            ->with('customer:id,profile_image_url')
            ->orderBy('test_date', 'desc')
            ->where(['customer_id' => $customer_id])
            ->first();
        
        if (!$customerStiTest) {
            return $this->errorResponse('No test data found.', 404);
        }
        
        return $this->successResponse(new QrCodeDataResource($customerStiTest), 'QR Code data fetched successfully.');
    }
    public function stiTestDetails(Request $request)
    {
        $customer_id = $request->customer_id;
        $testId = $request->test_id;
        $customer = Customer::select(
            'id',
            'first_name',
            'last_name',
            'profile_image_url',
            'email',
            'dob',
            'city',
            'gender_id',
            'email_verified_at',
            'mobile_verified'
        )
            ->with('gender:id,name')
            ->with([
                'labTestUsers' => function ($query) use ($testId) {
                    $query->select('lab_test_user_id as id', 'test_date', 'patient_id', 'customer_id', 'lab_id', 'lab_test_id')
                        ->where('lab_test_id', $testId);
                },
                'labTestUsers.lab' => function ($q) {
                    $q->select('id', 'name');
                }
            ])   // Load only id and name from genders table
            ->where('id', $customer_id)
            ->first();
        $labTests = LabTestUser::where('customer_id', $customer->id)->orderBy('test_date', 'DESC')->first();
        if ($labTests) {
            $testResult = StiTestResult::select('outcome')->where('lab_test_user_id', $labTests->lab_test_user_id)->first();
        }
        $customer->testStatus = isset($testResult->outcome) ? $testResult->outcome : 'positive';

        return $this->successResponse($customer, 'STI Test details fetched successfully.');
    }

    public function getStiTestResult(Request $request)
    {
        $customer_id = auth()->id();

        $result = DB::table('lab_test_results as r')
            ->select(
                'r.id',
                'r.result_document_url',
                'r.status',
                'r.outcome',
                'r.notes',
                'r.created_at',
                'r.updated_at',
                'ltu.lab_test_user_id',
                'labs.id as lab_id',
                'labs.name as lab_name',
                'customers.id as customer_id',
                 DB::raw("CONCAT(customers.first_name, ' ', customers.last_name) AS customer_name"),
                'customers.email as customer_email',
                'customers.profile_image_url as customer_avatar_url',
                'ltu.patient_id',
                'ltu.test_date',
                'lab_tests.id as lab_test_id',
                'lab_tests.name as test_name',
                'lab_tests.test_type',
                'tc.id as category_id',
                'tc.name as category_name'
            )
            ->leftJoin('lab_test_users as ltu', 'ltu.lab_test_user_id', '=', 'r.lab_test_user_id')
            ->leftJoin('labs', 'labs.id', '=', 'ltu.lab_id')
            ->leftJoin('customers', 'customers.id', '=', 'ltu.customer_id')
            ->leftJoin('lab_tests', 'lab_tests.id', '=', 'ltu.lab_test_id')
            ->leftJoin('test_categories as tc', 'tc.id', '=', 'lab_tests.category_id')
            ->whereNull('r.deleted_at')
            ->where('customers.id', $customer_id)
            ->orderBy('r.created_at', 'DESC')
            ->first();

        if (!$result) {
            return $this->errorResponse('No test result found.', 404);
        }

        return $this->successResponse(new StiTestResultResource($result), 'STI Test result fetched successfully.');
    }
}
